<?php
/**
 * Plugin Name: Snow Security Guard
 * Description: ログインID（ユーザー名）露出対策プラグイン（author 404 / REST users 制限 / XML-RPC 無効 等）
 * Version: 1.2.2
 * Author: Snow Tools
 * License: GPLv2 or later
 * Text Domain: snow-security-guard
 */

if (!defined('ABSPATH')) exit;

final class Snow_Security_Guard {

  const OPTION_KEY = 'snow_security_guard_options';

  public static function init() {
    add_action('admin_menu', [__CLASS__, 'add_settings_page']);
    add_action('admin_init', [__CLASS__, 'register_settings']);

    $opt = self::get_options();

    if (!empty($opt['author_404'])) {
      add_action('template_redirect', [__CLASS__, 'disable_author_archive_404'], 0);
      add_filter('author_link', [__CLASS__, 'override_author_link'], 10, 3);
      add_action('init', [__CLASS__, 'block_author_enumeration'], 0);
    }

    if (!empty($opt['rest_users'])) {
      add_filter('rest_endpoints', [__CLASS__, 'restrict_rest_users']);
    }

    if (!empty($opt['xmlrpc'])) {
      add_filter('xmlrpc_enabled', '__return_false');
    }

    if (!empty($opt['login_error'])) {
      add_filter('login_errors', [__CLASS__, 'hide_login_errors']);
    }
  }

  public static function defaults() {
    return [
      'author_404'  => 1,
      'rest_users'  => 1,
      'xmlrpc'      => 1,
      'login_error' => 1,
    ];
  }

  public static function get_options() {
    $raw = get_option(self::OPTION_KEY, []);
    if (!is_array($raw)) $raw = [];
    return wp_parse_args($raw, self::defaults());
  }

  public static function register_settings() {
    register_setting(
      'snow_security_guard',
      self::OPTION_KEY,
      [
        'sanitize_callback' => [__CLASS__, 'sanitize_options'],
        'default' => self::defaults(),
      ]
    );
  }

  /**
   * hiddenで 0 が常に来るので、'1' のときだけ 1 にする
   */
  public static function sanitize_options($input) {
    $defaults = self::defaults();
    $output = [];

    if (!is_array($input)) $input = [];

    foreach ($defaults as $key => $default) {
      $output[$key] = (isset($input[$key]) && (string)$input[$key] === '1') ? 1 : 0;
    }

    return $output;
  }

  public static function add_settings_page() {
    if (!current_user_can('manage_options')) return;

    add_options_page(
      'Snow Security Guard',
      'Snow Security Guard',
      'manage_options',
      'snow-security-guard',
      [__CLASS__, 'render_settings']
    );
  }

  public static function render_settings() {
    if (!current_user_can('manage_options')) wp_die('権限がありません。');

    $opt = self::get_options();
    ?>
    <div class="wrap">
      <h1>Snow Security Guard</h1>

      <form method="post" action="options.php">
        <?php settings_fields('snow_security_guard'); ?>

        <table class="form-table">

          <tr>
            <th>author アーカイブを 404</th>
            <td>
              <!-- ✅ OFFでも必ず0が送られる -->
              <input type="hidden" name="<?php echo esc_attr(self::OPTION_KEY); ?>[author_404]" value="0">
              <label>
                <input type="checkbox"
                       name="<?php echo esc_attr(self::OPTION_KEY); ?>[author_404]"
                       value="1"
                       <?php checked(!empty($opt['author_404'])); ?>>
                有効
              </label>
              <p class="description">
                投稿者ページ（/author/ユーザー名/）を存在しないページとして扱います。<br>
                ログインID（ユーザー名）がURLから推測されるのを防ぐための対策です。<br>
                通常のブログ運用では影響はありません。
              </p>
            </td>
          </tr>

          <tr>
            <th>REST users 制限</th>
            <td>
              <input type="hidden" name="<?php echo esc_attr(self::OPTION_KEY); ?>[rest_users]" value="0">
              <label>
                <input type="checkbox"
                       name="<?php echo esc_attr(self::OPTION_KEY); ?>[rest_users]"
                       value="1"
                       <?php checked(!empty($opt['rest_users'])); ?>>
                有効
              </label>
              <p class="description">
                REST API（/wp-json/wp/v2/users）を未ログイン状態では利用できなくします。<br>
                ユーザー名の取得や列挙を防ぐ目的です。<br>
                外部サービスや独自API連携でユーザー情報を取得している場合は OFF にしてください。
              </p>
            </td>
          </tr>

          <tr>
            <th>XML-RPC 無効化</th>
            <td>
              <input type="hidden" name="<?php echo esc_attr(self::OPTION_KEY); ?>[xmlrpc]" value="0">
              <label>
                <input type="checkbox"
                       name="<?php echo esc_attr(self::OPTION_KEY); ?>[xmlrpc]"
                       value="1"
                       <?php checked(!empty($opt['xmlrpc'])); ?>>
                有効
              </label>
              <p class="description">
                XML-RPC を無効にします。<br>
                ブルートフォース攻撃に利用されやすいため、不要であれば無効化を推奨します。<br>
                Jetpackや一部の外部投稿ツールを使用している場合は OFF にしてください。
              </p>
            </td>
          </tr>

          <tr>
            <th>ログインエラー統一</th>
            <td>
              <input type="hidden" name="<?php echo esc_attr(self::OPTION_KEY); ?>[login_error]" value="0">
              <label>
                <input type="checkbox"
                       name="<?php echo esc_attr(self::OPTION_KEY); ?>[login_error]"
                       value="1"
                       <?php checked(!empty($opt['login_error'])); ?>>
                有効
              </label>
              <p class="description">
                ログイン失敗時のエラーメッセージを統一します。<br>
                「ユーザー名が存在するかどうか」を推測されにくくするための対策です。<br>
                ログイン動作自体には影響しません。
              </p>
            </td>
          </tr>

        </table>

        <?php submit_button(); ?>
      </form>
    </div>
    <?php
  }

  public static function disable_author_archive_404() {
    if (is_author()) {
      global $wp_query;
      $wp_query->set_404();
      status_header(404);
      nocache_headers();

      $tpl = get_query_template('404');
      if ($tpl && file_exists($tpl)) {
        include $tpl;
      } else {
        echo '404 Not Found';
      }
      exit;
    }
  }

  public static function override_author_link($link, $author_id, $author_nicename) {
    return home_url('/');
  }

  public static function restrict_rest_users($endpoints) {
    if (!is_user_logged_in()) {
      unset($endpoints['/wp/v2/users']);
      unset($endpoints['/wp/v2/users/(?P<id>[\\d]+)']);
    }
    return $endpoints;
  }

  public static function hide_login_errors($error = '') {
    return 'ログイン情報が正しくありません。';
  }

  public static function block_author_enumeration() {
    if (is_admin()) return;

    if (isset($_GET['author'])) {
      wp_redirect(home_url('/'), 302);
      exit;
    }
  }
}

Snow_Security_Guard::init();
